"use client";

import { useState, useEffect, useCallback } from "react";
import GaugeChart from "../../../components/GaugeChart";
import NotificationsTab from "../../../user-panel/NotificationsTab";
import { useToast } from "../../../components/ui/use-toast";
import { ToastAction } from "../../../components/ui/toast";

interface DeviceData {
  devEUI: string;
  // Add other device data properties as needed
}

export default function DevicePage({ params }: { params: { deviceId: string } }) {
  const [deviceData, setDeviceData] = useState<DeviceData | null>(null);
  const [currentValue, setCurrentValue] = useState<number | null>(null);
  const [maxValue, setMaxValue] = useState<number | null>(null);
  const [isLoading, setIsLoading] = useState(true);
  const [commandFlags, setCommandFlags] = useState<Record<string, boolean>>({});
  const { toast } = useToast();

  useEffect(() => {
    // Fetch device data
    const fetchDeviceData = async () => {
      try {
        const response = await fetch(`/api/devices/${params.deviceId}`);
        if (!response.ok) throw new Error('Failed to fetch device data');
        const data = await response.json();
        setDeviceData(data);
      } catch (error) {
        console.error('Error fetching device data:', error);
      }
    };

    fetchDeviceData();
  }, [params.deviceId]);

  const handleThresholdReached = useCallback((deviceId: string, message: string, thresholdKey: string) => {
    console.log('DevicePage: Threshold reached:', { deviceId, message, thresholdKey }); // Debug log

    // Show toast notification
    toast({
      variant: thresholdKey === "100%" ? "destructive" : "default",
      title: `Threshold Alert: ${thresholdKey}`,
      description: message,
      action: (
        <ToastAction altText="View Details" onClick={() => {
          // Scroll to notifications tab
          const notificationsTab = document.getElementById('notifications-tab');
          if (notificationsTab) {
            notificationsTab.scrollIntoView({ behavior: 'smooth' });
          }
        }}>
          View Details
        </ToastAction>
      ),
    });

    // Call the window handler if it exists
    if ((window as any).handleThreshold) {
      console.log('Calling window.handleThreshold');
      (window as any).handleThreshold(deviceId, message, thresholdKey);
    } else {
      console.warn('window.handleThreshold not found');
    }
  }, [toast]);

  // Debug log for component render
  console.log('DevicePage render:', {
    deviceId: params.deviceId,
    currentValue,
    maxValue,
    isLoading,
    commandFlags,
    hasCallback: !!handleThresholdReached
  });

  return (
    <div className="container mx-auto p-4 space-y-4">
      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <GaugeChart
          deviceName={params.deviceId}
          devEUI={deviceData?.devEUI || ""}
        />
        <div id="notifications-tab">
          <NotificationsTab isSystem={false} />
        </div>
      </div>
    </div>
  );
} 